
import os
import pandas as pd
from q5_comp_lib import GridSpec, InstrumentSpec, SweepSpec, Flags, run_profiles_and_summary, write_manifest_yaml, write_audit_json

def main():
    grid = GridSpec(
        Lx=509, Ly=481, screen_x=508,
        slit_y_up=+10, slit_y_dn=-10,
        band_increments=(5,9,13,17,21),
        band_height=20,
        slant_alpha=0.0
    )
    instr = InstrumentSpec(
        M=24, jitter_halfwidth=0.15, epsilon=0.10,
        roi_halfwidth=80, smooth_box_w=9
    )
    sweep = SweepSpec(
        gravities=(1,0,2),
        s_ww_values=(0.00,0.25,0.50,0.75,0.90),
        seeds=(101,202,303),
        K_profiles=8_000, K_summary=50_000
    )
    flags = Flags(
        use_numba=True, use_dijkstra_4nbr=False,
        keep_profiles_full_range=True, analytic_fast_mode=False
    )

    outdir = os.path.abspath("./q5_out")
    os.makedirs(outdir, exist_ok=True)
    print(">> Q5 run starting (minimal run order)...")
    profiles_df, summary_df, audit = run_profiles_and_summary(grid, instr, sweep, flags)

    profiles_csv = os.path.join(outdir, "q5_comp_profiles.csv")
    summary_csv  = os.path.join(outdir, "q5_comp_summary.csv")
    manifest_yaml= os.path.join(outdir, "q5_comp_manifest.yaml")
    audit_json   = os.path.join(outdir, "q5_comp_audit.json")

    profiles_df.to_csv(profiles_csv, index=False)
    summary_df.to_csv(summary_csv, index=False)
    write_manifest_yaml(manifest_yaml, grid, instr, sweep, flags)
    write_audit_json(audit_json, audit)

    print("\n== V(D) per gravity (in s_ww order) ==")
    for g in summary_df["g"].unique():
        sub = summary_df[summary_df["g"]==g].sort_values("s_ww")
        pairs = list(zip(sub["D"].round(4).tolist(), sub["V"].round(4).tolist()))
        mono = audit["curve_lint"][str(g)]["monotone_nonincreasing_V_vs_sww"]
        print(f"g={g}: {pairs}  | monotone={mono}")

    print("\nArtifacts written to:", outdir)
    print("PASS/FAIL guardrails summary (see audit.json for details):")
    print(f"- curve_lint: {all(v['monotone_nonincreasing_V_vs_sww'] for v in audit['curve_lint'].values())}")
    print(f"- no_skip: {audit['no_skip']}")
    print(f"- pf_born_ties_only: {audit['pf_born_ties_only']} (born_invocations={audit['born_invocations']})")
    print(f"- Englert max excess per g:", {g: v["max_excess"] for g, v in audit["englert_check"].items()})

if __name__ == "__main__":
    main()
